#dependent on Logger module
function Wait-Network {
    while (((Test-NetConnection -InformationLevel Quiet) -or (test-connection 1.1.1.1)) -eq $false) {
        write-host 'Waiting for Network'; 
        Start-Sleep 1;
    }
}

function Download-File {
    param ( [Parameter(Mandatory = $true)] [string] $url,
        [Parameter(Mandatory = $true)] [string] $outFilePath,
        [Parameter(Mandatory = $true)] $logger,
        [Parameter(Mandatory = $false)] [int] $numOfRetries = 3
    )

    $attemptCount = 0
    $completed = $false

    while ( ($attemptCount -lt $numOfRetries) -and (-Not $completed)) {       
        $attemptCount++
        try {
            $parentPath = split-path -Parent $outFilePath
            if (-Not (Test-Path $parentPath)) {
                New-Item $parentPath -type directory -Force
            }
            [string]$logMsgCommonPart = "file from $url into $outFilePath,  attempt number $attemptCount"
            if (Test-Path $outFilePath) {
                Remove-Item -Path $outFilePath
            }
            $logger.Info("Before downloading $logMsgCommonPart")
            
            #Timeout should be added to here. Response timeout are indefinate
            $wc = New-Object System.Net.WebClient
            $wc.DownloadFile($url,$outFilePath)
            $logger.Info("After downloading $logMsgCommonPart")
            $completed = $true
        } catch {
            $logger.Error("Could not download $logMsgCommonPart. Exception Message: $($_.Exception.Message)")
            if ($ErrorActionPreference -eq "Stop" -and ($attemptCount -ge $numOfRetries)) {
                throw
            }
        }
    }
}

<#
.SYNOPSIS
Downloads and unzips a .zip file into a specified directory

.DESCRIPTION
Long description

.PARAMETER url
Parameter description

.PARAMETER folderPath
Parameter description

.PARAMETER logger
Parameter description

.EXAMPLE
An example

.NOTES
General notes
#>
function Download-Unzip {
    param(
        [Parameter(Mandatory = $true)] [string] $url,
        [Parameter(Mandatory = $true)] [string] $folderPath,
        [Parameter(Mandatory = $true)] $logger,
        [Parameter(Mandatory = $false)] [int] $numOfRetries = 1
    )

    $folderName = Split-Path -Leaf $folderPath
    $zipFilePath = "$Env:Temp\$folderName.zip"

    try {
        $parentPath = Split-Path -Parent $folderPath
        if (-Not (Test-Path $parentPath)) {
            New-Item $parentPath -type directory -Force
        }
        Download-File $url $zipFilePath $logger $numOfRetries
        Expand-Archive -Path $zipFilePath -DestinationPath $folderPath -Force -ErrorAction Stop
        Remove-Item -Path $zipFilePath -Force
    }
    catch {
        $logger.Error("Could not extract from $zipFilePath into $folderPath. Exception Message: $($_.Exception.Message)")
        if ($ErrorActionPreference -eq "Stop") {
            throw
        }
    }
}

function Kill-ScheduledTask {
    param(
        [Parameter(Mandatory = $true)][string] $taskname,
        [Parameter(Mandatory = $true)] $logger,
        [Parameter(Mandatory = $false)][int] $timeout = 0
    )
 
    try {
        $killer = Start-Job -Name "Wait for Scheduled Task: $taskname" {
            While ( -Not (Get-ScheduledTask -ErrorAction Stop | Where-Object Taskname -match $taskname)) { Start-Sleep 5; }
        }
    
        <#
            Wait for task to appear.
            Timeout for wait is 10 mins
        #>
        $waitedUponJob = $killer | Wait-Job -Timeout $timeout
        if ($waitedUponJob -eq $null) {
            #timed out, no matching task found
            $logger.Warn("In $($MyInvocation.MyCommand): Scheduled Task: $taskname not found")
            return $false
        }
        $logger.Info("In $($MyInvocation.MyCommand): before disabling $taskname task")
        $tasks = Get-ScheduledTask -ErrorAction Stop | Where-Object Taskname -match $taskname
        if ($tasks.Count -gt 0) {
            $task | Disable-ScheduledTask -ErrorAction Stop
            $logger.Info("In $($MyInvocation.MyCommand): after disabling $taskname task")
            return $true
        }
        $logger.Warn("In $($MyInvocation.MyCommand): Scheduled Task: $taskname not found")
        return $false
    }
    catch {
        $logger.Error("In $($MyInvocation.MyCommand): could not disable $taskname task. Exception msg: $($_.Exception.Message)")

        if ($ErrorActionPreference -eq "Stop") {
            throw
        }
    }
}


function Run-Msi {
    param ( [Parameter(Mandatory = $true)] [string] $programName,
        [Parameter(Mandatory = $true)] [string] $path,
        [Parameter(Mandatory = $true)] $logger,
        [Parameter(Mandatory = $true)] [string] $arg1,
        [Parameter(Mandatory = $false)] [string] $arg2
    ) 
    

    $logger.Info("Before installing $programName")
    msiexec /i $path $arg1 $arg2 | Out-Null
    $successLogMessage = "After installing $programName from $path"
    switch ($LASTEXITCODE) {
        0 { $logger.Info($successLogMessage); Break; }
        1641 { $logger.Info("$successLogMessage. Rebooting."); Break; }
        3010 { $logger.Info("$successLogMessage. Reboot required."); Break; }
        default {
            $errmsg = "msiexec failed to install $programName from $path. Exit code: $lastexitcode"
            $logger.Error($errmsg);
    
            if ($ErrorActionPreference -eq "Stop") {
                throw [System.Exception] $errmsg
            }
        }
    }

}

function Add-RegistryKeys
 {
    param (
        [Parameter(Mandatory = $true)] $paths,
        [Parameter(Mandatory = $true)] $logger
    )

    foreach ($path in $paths) {
        Add-RegistryKey $path $logger
    }
}

function Add-RegistryKey
{
   param (
       [Parameter(Mandatory = $true)] $path,
       [Parameter(Mandatory = $true)] $logger
   )
       if(Test-path $path) {
           $logger.Info("$path already exists.")
       } else {
           $logger.Info("Before creating $path")
           New-Item -Path $path -Force
           $logger.Info("After creating $path")
       }
}

Export-ModuleMember -Function Wait-Network
Export-ModuleMember -Function Download-Unzip
Export-ModuleMember -Function Download-File
Export-ModuleMember -Function Kill-ScheduledTask
Export-ModuleMember -Function Run-Msi
Export-ModuleMember -Function Add-RegistryKeys
Export-ModuleMember -Function Add-RegistryKey


