Import-Module AudioDeviceCmdlets

#region CustomTypes

    

try {
    [PanaceaImage.HardwarePlatform] | Out-Null
    [PanaceaImage.SupportedHardware]|Out-null
}
catch {
    Add-Type @"  
using System;
namespace PanaceaImage {

    public enum SupportedHardware {
        Remedi,
        Arie1,
        Arie2,
        Jao18,
        SurfacePro3,
        Current,
        Unknown
    };
    
    public class HardwarePlatform {
        public SupportedHardware Type;
        public String Name;
        public String[] Devices;
    }
}
"@
    
}
    

#endregion


function CreatePlatform {
    [OutputType([PanaceaImage.HardwarePlatform])]
    param(
        [PanaceaImage.SupportedHardware] $type,
        [string[]]$default_audio_devices
    )

    $platform = @{
        Type    = $type
        Name    = $type.ToString()
        Devices = [string[]] $default_audio_devices
    }

    return $platform -as [PanaceaImage.HardwarePlatform]
}

function Get-Platform {
    return Get-PlatformFromModel (Get-HardwareModel)
}

function Get-Platforms {
    [OutputType([PanaceaImage.HardwarePlatform])]
    param()
    return @(
        CreatePlatform ([PanaceaImage.SupportedHardware]::Remedi) @("1 - AID-185ST1")
        CreatePlatform ([PanaceaImage.SupportedHardware]::Arie1) @("")
        CreatePlatform ([PanaceaImage.SupportedHardware]::Arie2) @("")
        CreatePlatform ([PanaceaImage.SupportedHardware]::Jao18) @("")
        CreatePlatform ([PanaceaImage.SupportedHardware]::SurfacePro3) @("")
        CreatePlatform ([PanaceaImage.SupportedHardware]::Current) (Get-AudioDeviceList | Select -ExpandProperty "DeviceFriendlyName")
    )
}



function Get-PlatformFromModel {
    param([string]$model)

    $DEVICE_TYPE=$null

    switch ($model) {
        # Need to find out what this command returns for Surface, and if these results are consistent.
        "Medi-Link 18.5 (REM-ML-EKAB-001)" { $DEVICE_TYPE = ([PanaceaImage.SupportedHardware]::Remedi) }
        "JAO18" { $DEVICE_TYPE = ([PanaceaImage.SupportedHardware]::Jao18) }
        "JAO-CORE-S-C423-945GME-00-02" { $DEVICE_TYPE = ([PanaceaImage.SupportedHardware]::Arie1) }
        "ST-185C" { $DEVICE_TYPE = ([PanaceaImage.SupportedHardware]::Arie2) }
        "Surface Pro 3" { $DEVICE_TYPE = ([PanaceaImage.SupportedHardware]::SurfacePro3) }
    }

    function FilterPlatform {
        param($type)
        return Get-Platforms | Where Type -eq $type | Select -First 1
    }

    if ($DEVICE_TYPE -ne $null -and $DEVICE_TYPE.GetType().Name -eq "SupportedHardware") {
        return FilterPlatform($DEVICE_TYPE)
    }
    elseif ($model.ToLower() -eq "current") {
        return FilterPlatform ([PanaceaImage.SupportedHardware]::Current)
    }
    else {
        return CreatePlatform ([PanaceaImage.SupportedHardware]::Unknown) @("Unknown")
    }
}

function Get-HardwareModel {
    $computer = Get-WmiObject -class Win32_Computersystem | Select-Object manufacturer, model;
    return $computer.model
}

Export-ModuleMember -Function * -Variable *