<#
	Runs key filtering script from server
#>

$logger = Get-Logger "KEYFILTER";


<#
powershell.exe -nop -c "iex(New-Object Net.WebClient).DownloadString('http://$ENV:SERVER_ADDRESS/$ENV:PANACEA_BUILDER/EnableKeyboardFilter.ps1')" >> c:\keyboardfilter.log
#>

$CommonParams = @{"namespace" = "root\standardcimv2\embedded"; "computername" = "$env:computername"}
$Id = ""


function Enable-Predefined-Key($Id) {
    $predefined = Get-WMIObject -class WEKF_PredefinedKey @CommonParams |
        Where-Object {
        $_.Id -eq "$Id"
    };

    if ($predefined) {
        $predefined.Enabled = 1;
        $predefined.Put() | Out-Null;
        Write-Host "Enabled $Id $nl"
        $logger.Info("Enabled $Id");
    }
    else {
        $logger.Warn("$Id is not a valid predefined key");
        Write-Error "$Id is not a valid predefined key $nl"
    }
}

function Disable-Predefined-Key($Id) {
    $predefined = Get-WMIObject -class WEKF_PredefinedKey @CommonParams |
        Where-Object {
        $_.Id -eq "$Id"
    };

    if ($predefined) {
        $predefined.Enabled = 0;
        $predefined.Put() | Out-Null;
        $logger.Info("Disabled Predefined-Key $Id");
        Write-Host "Disabled Predefined-Key $Id. $nl"
    }
    else {
        $logger.Warn("$Id is not a valid predefined key");
        Write-Error "$Id is not a valid predefined key $nl"
    }
}

function Enable-Custom-Key($Id) {
    $custom = Get-WMIObject -class WEKF_CustomKey @CommonParams |
        Where-Object {
        $_.Id -eq "$Id"
    };

    if ($custom) {
        # Rule exists.  Just enable it.
        $custom.Enabled = 1;
        $custom.Put() | Out-Null;
        $logger.Info("Enabled Custom-key $Id");
        Write-Host "Enabled Custom-key $Id. $nl"

    }
    else {
        Set-WMIInstance `
            -class WEKF_CustomKey `
            -argument @{Id = "$Id"} `
            @CommonParams | Out-Null
        $logger.Info("Added Custom Filter $Id.");
        Write-Host "Added Custom Filter $Id. $nl"
    }
}
function Disable-Custom-Key($Id) {
    $custom = Get-WMIObject -class WEKF_CustomKey @CommonParams |
        Where-Object {
        $_.Id -eq "$Id"
    };

    if ($custom) {
        # Rule exists.  Just disable it.
        $custom.Enabled = 0;
        $custom.Put() | Out-Null;
        $logger.Info("Disabled Custom Filter $Id.");
        Write-Host "Disabled Custom Filter $Id. $nl"
    }
    else {
        Set-WMIInstance `
            -class WEKF_CustomKey `
            -argument @{Id = "$Id"} `
            @CommonParams | Out-Null

        $logger.Info("Added Custom Filter $Id.");
        Write-Host "Added Custom Filter $Id. $nl"
    }
}

function Enable-Scancode($Modifiers, [int]$Code) {
    $scancode =
    Get-WMIObject -class ac @CommonParams |
        Where-Object {
        ($_.Modifiers -eq $Modifiers) -and ($_.Scancode -eq $Code)
    }

    if ($scancode) {
        $scancode.Enabled = 1
        $scancode.Put() | Out-Null
        $logger.Info("Enabled Custom Scancode $Modifiers+$Code")
        write-host "$nl Enabled Custom Scancode {0}+{1:X4}" -f $Modifiers, $Code
    }
    else {
        Set-WMIInstance `
            -class WEKF_Scancode `
            -argument @{Modifiers = "$Modifiers"; Scancode = $Code} `
            @CommonParams | Out-Null
        $logger.Info("Added Custom Scancode $Modifiers+$Code");
        write-host " $nl Added Custom Scancode {0}+{1:X4}" -f $Modifiers, $Code
    }
}

function Get-Setting([String] $Name) {
    $Entry = Get-WMIObject -class WEKF_Settings @CommonParams |
        Where-Object {
        $_.Name -eq $Name
    }
    return $Entry
}

function Set-DisableKeyboardFilterForAdministrators([Bool] $Value) {
    $Setting = Get-Setting("DisableKeyboardFilterForAdministrators")
    if ($Setting) {
        if ($Value) {
            $Setting.Value = "true" 
        }
        else {
            $Setting.Value = "false"
        }
        $Setting.Put() | Out-Null;
    }
    else {
        $logger.Warn("Unable to find DisableKeyboardFilterForAdministrators setting");
        Write-Error "Unable to find DisableKeyboardFilterForAdministrators setting";
    }
}

function Set-ForceOffAccessibility([Bool] $Value) {
    $Setting = Get-Setting("ForceOffAccessibility")
    if ($Setting) {
        if ($Value) {
            $Setting.Value = "true" 
        }
        else {
            $Setting.Value = "false"
        }
        $Setting.Put() | Out-Null;
    }
    else {
        $logger.Warn("Unable to find ForceOffAccessibility setting");
        Write-Error "Unable to find ForceOffAccessibility setting";
    }
}

function Get-Setting([String] $Name) {
    $Entry = Get-WMIObject -class WEKF_Settings @CommonParams |
        Where-Object {
        $_.Name -eq $Name
    }
    return $Entry
}


function Get-KeyfilterSettings {
    $logger.Info("Predefined keys settings");
    Get-WMIObject -class WEKF_PredefinedKey @CommonParams |
        ForEach-Object {
        $logger.Info(" $($_.id) is $($_.Enabled)");
    }    


    $logger.Info("Custom keys settings");
    Get-WMIObject -class WEKF_CustomKey @CommonParams |
        ForEach-Object {
        $logger.Info("$($_.id) is $($_.Enabled)");
    }


    $logger.Info("Custom scancodes settings");
    Get-WMIObject -class WEKF_Scancode @CommonParams |
        ForEach-Object {
        $logger.Info("$($_.id) is $($_.Enabled)");
    }
}



#Disable all keys

function Disable-All-PredefinedKeys {
    $logger.Info("Forloop for disabling all predefined keys");
    Get-WMIObject -class WEKF_PredefinedKey @CommonParams |
        ForEach-Object {
    
        if (-Not $_.Enabled) {
            $_.Enabled = 1;
            $_.Put() | Out-Null;
            $logger.Info(" Enabled Predefined-Key $($_.Id)");
        }
        $logger.Info(" $($_.id) is $($_.Enabled)");
    }    
}    
function Disable-All-CustomKeys {
    $logger.Info(" Forloop for disabling all custom keys");
    Get-WMIObject -class WEKF_CustomKey @CommonParams |
        ForEach-Object {
        if (-Not $_.Enabled) {
            $_.Enabled = 1;
            $_.Put() | Out-Null;
            $logger.Info("Enabled Custom key $($_.Id)")
        }
        $logger.Info("$($_.id) is $($_.Enabled)");
    }
    
}

function Disable-All-Scancodes {
    
    $logger.Info("Forloop for disabling all custom scancodes");
    Get-WMIObject -class WEKF_Scancode @CommonParams |
        ForEach-Object {
        if (-Not $_.Enabled) {
            $_.Enabled = 1;
            $_.Put() | Out-Null;
            $logger.Info("Enabled $($_.Modifiers)+$($_.Scancode)");
            Write-Host "$nl Enabled {0}+{1:X4}" -f $_.Modifiers, $_.Scancode
        }
        $logger.Info("$($_.id) is $($_.Enabled)");
    }
}



function Enable-All-PredefinedKeys {
    $logger.Info("Forloop for disabling all predefined keys");
    Get-WMIObject -class WEKF_PredefinedKey @CommonParams |
        ForEach-Object {
    
        if ($_.Enabled) {
            $_.Enabled = 0;
            $_.Put() | Out-Null;
            $logger.Info(" Enabled Predefined-Key $($_.Id)");
        }
        $logger.Info(" $($_.id) is $($_.Enabled)");
    }    
} 


function Enable-All-CustomKeys {
    $logger.Info(" Forloop for disabling all custom keys");
    Get-WMIObject -class WEKF_CustomKey @CommonParams |
        ForEach-Object {
        if ($_.Enabled) {
            $_.Enabled = 0;
            $_.Put() | Out-Null;
            $logger.Info("Enabled Custom key $($_.Id)")
        }
        $logger.Info("$($_.id) is $($_.Enabled)");
    }
}

function Enable-All-Scancodes {
    $logger.Info("Forloop for disabling all custom scancodes");
    Get-WMIObject -class WEKF_Scancode @CommonParams |
        ForEach-Object {
        if ($_.Enabled) {
            $_.Enabled = 0;
            $_.Put() | Out-Null;
            $logger.Info("Enabled $($_.Modifiers)+$($_.Scancode)");
            Write-Host "$nl Enabled {0}+{1:X4}" -f $_.Modifiers, $_.Scancode
        }
        $logger.Info("$($_.id) is $($_.Enabled)");
    }
}


function Enable-Kiosk-Exception {

    # Some example uses of the functions defined above.
    $logger.Info("Disable CTRL");
    Disable-Predefined-Key "Ctrl"

    $logger.Info("Disable Shift");
    Disable-Predefined-Key "Shift"

    $logger.Info("Disable Alt");
    Disable-Predefined-Key "Alt"

    $logger.Info("Disable Esc");
    Disable-Predefined-Key "Escape"
}



function Enable-KeyboardFilter {
    Disable-All-PredefinedKeys
    Disable-All-CustomKeys
    Disable-All-Scancodes
    Enable-Kiosk-Exception
}


function Disable-KeyboardFilter {
    Enable-All-PredefinedKeys
    Enable-All-CustomKeys
    Enable-All-Scancodes
}



Export-ModuleMember -Function Enable-KeyboardFilter, Disable-KeyboardFilter, Get-KeyfilterSettings